/******************************************************************************/
/* OpenSi : Outils libres de gestion d'entreprise                             */
/* Copyright (C) 2003 Speedinfo.fr S.A.R.L.                                   */
/* Contact: contact@opensi.org                                                */
/*                                                                            */
/* This program is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU General Public License                */
/* as published by the Free Software Foundation; either version 2             */
/* of the License, or (at your option) any later version.                     */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/******************************************************************************/


package org.opensi.facturation.reglements.clients;

import java.io.IOException;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;

import org.opensi.util.pdf.MultiPagesDocument;
import org.opensi.util.pdf.TableBuilder;

import com.lowagie.text.DocumentException;
import com.lowagie.text.Font;
import com.lowagie.text.Image;
import com.lowagie.text.Phrase;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;


public class PdfRemboursement extends MultiPagesDocument {


	private static final Font fontAdrFacture = new Font(Font.HELVETICA, 11, Font.BOLD);
	private static final Font fontAdrSociete = new Font(Font.HELVETICA, 10, Font.NORMAL);
	private static final Font fontAdrSocieteGras = new Font(Font.HELVETICA, 11, Font.BOLD);
	private static final Font fontWeb = new Font(Font.HELVETICA, 10, Font.ITALIC);
	private static final Font fontLegale = new Font(Font.HELVETICA , 8, Font.NORMAL);
	private static final Font fontTexte = new Font(Font.HELVETICA, 11, Font.NORMAL);

	private DecimalFormat numFormat = new DecimalFormat("0.00");
	private DecimalFormat numFormatCap = new DecimalFormat("0.##");


	private DataRemboursement dr;
	private Image logo;


	public PdfRemboursement(String tempdir, DataRemboursement dr) throws Exception {
		
		super(tempdir);
		
		this.dr = dr;


		DecimalFormatSymbols dfs = new DecimalFormatSymbols();
		dfs.setDecimalSeparator('.');
		dfs.setGroupingSeparator(' ');

		numFormat.setDecimalFormatSymbols(dfs);
		numFormat.setNegativeSuffix("-");
		numFormat.setNegativePrefix("");
		numFormat.setPositiveSuffix(" ");
		numFormat.setGroupingUsed(true);
		numFormat.setGroupingSize(3);

		numFormatCap.setDecimalFormatSymbols(dfs);
		numFormatCap.setGroupingUsed(true);
		numFormatCap.setGroupingSize(3);
		
		try {
			logo = Image.getInstance(dr.urlLogo);
			if (logo.getDpiX()!=0 && logo.getDpiY()!=0)
				logo.scaleAbsolute(logo.getPlainWidth() * 72 / logo.getDpiX(), logo.getPlainHeight() * 72 / logo.getDpiY());
		}
		catch (IOException e) {
			logo = null;
		}

	}
	

/* implmentation des fonctions de MultiPagesDocument */
	
	public PdfPTable firstHeader() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		PdfPTable pptHeader = new PdfPTable(1);
		pptHeader.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(makeEntete());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPadding(0);
		pptHeader.addCell(cell);

		cell = new PdfPCell(makeSpace(50));
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPadding(0);
		pptHeader.addCell(cell);

		tb.add(pptHeader);
		
		return tb.getTable();	
	}
	
	
	public PdfPTable middleHeader(int lastLine) throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(firstHeader());
		
		return tb.getTable();
	}
	
	
	public PdfPTable lastFooter() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeFinPage());
		tb.add(makeInfosLegales());
		
		return tb.getTable();
	}
	
	
	public PdfPTable middleFooter(int nbLinesPage, int lastLine) throws DocumentException {
			
		TableBuilder tb = new TableBuilder();
		
		tb.add(lastFooter());
		
		return tb.getTable();		
	}
	
	
	public PdfPTable bodyLine(int numLine) throws DocumentException {
	
		return makeLettre();
	}
	
	
	public PdfPTable emptyLine(float emptyHeight) throws DocumentException {
		
		PdfPTable pptSpace = new PdfPTable(1);
		pptSpace.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(emptyHeight);		
		cell.setPhrase(new Phrase(""));		
		pptSpace.addCell(cell);
	
		return pptSpace;
	}
	
	
	public int nbLines() {
		return 1;
	}
	
	
	public int nbEx() {
		return 1;
	}
	
	
	public Image imageFond() {
		return null;
	}



	/* mthodes internes */


	private PdfPTable makeEntete() throws DocumentException {
		
		PdfPTable pptEntete = new PdfPTable(2);
		pptEntete.setWidthPercentage(100);
		int[] widths = {50,50};
		pptEntete.setWidths(widths);
		
		PdfPTable pptEnteteGauche = new PdfPTable(1);
		pptEnteteGauche.setWidthPercentage(100);
		PdfPTable pptEnteteDroite = new PdfPTable(1);
		pptEnteteDroite.setWidthPercentage(100);
		
		PdfPCell cellSociete = new PdfPCell(makeInfosSociete());
		cellSociete.setMinimumHeight(100);
		cellSociete.setBorder(PdfPCell.NO_BORDER);
		pptEnteteGauche.addCell(cellSociete);
		PdfPCell cellVide = new PdfPCell(pptEnteteGauche.getDefaultCell());
		cellVide.setBorder(PdfPCell.NO_BORDER);
		cellVide.setPhrase(new Phrase(""));
		pptEnteteGauche.addCell(cellVide);
		
		cellVide.setFixedHeight(100);
		pptEnteteDroite.addCell(cellVide);		
		PdfPCell cellAdrEnvoi = new PdfPCell(makeAdrEnvoi());
		cellAdrEnvoi.setMinimumHeight(80);
		cellAdrEnvoi.setBorder(PdfPCell.NO_BORDER);
		pptEnteteDroite.addCell(cellAdrEnvoi);
		pptEnteteDroite.addCell(cellVide);
		
		PdfPCell cellPartieGauche = new PdfPCell(pptEnteteGauche);
		cellPartieGauche.setBorder(PdfPCell.NO_BORDER);
		PdfPCell cellPartieDroite = new PdfPCell(pptEnteteDroite);		
		cellPartieDroite.setBorder(PdfPCell.NO_BORDER);
		pptEntete.addCell(cellPartieGauche);
		pptEntete.addCell(cellPartieDroite);
		
		return pptEntete;
	}
	
	
	private PdfPTable makeFinPage() {
		
		PdfPTable pptFP = new PdfPTable(1);
		pptFP.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptFP.getDefaultCell());
		cell.setBorder(PdfPCell.TOP);
		cell.setPhrase(new Phrase(""));
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(1);
				
		pptFP.addCell(cell);
	
		return pptFP;
	}


	private PdfPTable makeSpace(int space) throws DocumentException {

		PdfPTable pptSpace = new PdfPTable(1);

		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setFixedHeight(space);
		cell.setPhrase(new Phrase(""));

		pptSpace.addCell(cell);

		return pptSpace;
	}


	private PdfPTable makeInfosSociete() {

		PdfPTable pptInfos = new PdfPTable(1);

		PdfPCell cell;
		
		if (logo != null) {
			cell = new PdfPCell(logo, false);

			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

			pptInfos.addCell(cell);

			cell.setImage(null);

			cell.setMinimumHeight(5);
			cell.setPhrase(new Phrase("", fontAdrSociete));
			pptInfos.addCell(cell);
			cell.setMinimumHeight(0);
		}
		else {
			cell = new PdfPCell(pptInfos.getDefaultCell());
			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		}

		if (!dr.logoAdr) {
			cell.setPhrase(new Phrase(dr.raisonSociale, fontAdrSocieteGras));
			pptInfos.addCell(cell);
			cell.setPhrase(new Phrase(dr.adresse1, fontAdrSociete));
			pptInfos.addCell(cell);
			if (!dr.adresse2.equals("")) {
				cell.setPhrase(new Phrase(dr.adresse2, fontAdrSociete));
				pptInfos.addCell(cell);
			}
			if (!dr.adresse3.equals("")) {
				cell.setPhrase(new Phrase(dr.adresse3, fontAdrSociete));
				pptInfos.addCell(cell);
			}
			cell.setPhrase(new Phrase(dr.codePostal +" "+ dr.ville, fontAdrSociete));
			pptInfos.addCell(cell);

			String telfax = "";

			if (dr.tel.length()>0) {
				telfax = "T\u00E9l: "+ dr.tel;
				if (dr.fax.length()>0) {
					telfax += " - Fax: "+ dr.fax;
				}
			}
			else if (dr.fax.length()>0) {
				telfax = "Fax: "+ dr.fax;
			}

			if (telfax.length()>0) {
				cell.setPhrase(new Phrase(telfax, fontAdrSociete));
				pptInfos.addCell(cell);
			}

			if (!dr.email.equals("")) {
				cell.setPhrase(new Phrase("e-mail: "+ dr.email, fontAdrSociete));
				pptInfos.addCell(cell);
			}
			if (!dr.web.equals("")) {
				cell.setPhrase(new Phrase(dr.web, fontWeb));
				pptInfos.addCell(cell);
			}
		}

		cell.setPhrase(new Phrase(""));
		pptInfos.addCell(cell);

		return pptInfos;
	}


	private PdfPTable makeAdrEnvoi() {

		PdfPTable pptInfosFacture = new PdfPTable(1);
		PdfPCell cell = new PdfPCell(pptInfosFacture.getDefaultCell());

		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setPaddingBottom(4);
		cell.setBorder(PdfPCell.NO_BORDER);

		cell.setPhrase(new Phrase(dr.denominationDest, fontAdrFacture));
		pptInfosFacture.addCell(cell);

		cell.setPhrase(new Phrase(dr.adresse1Dest, fontAdrFacture));
		pptInfosFacture.addCell(cell);

		if (!dr.adresse2Dest.equals("")) {
			cell.setPhrase(new Phrase(dr.adresse2Dest, fontAdrFacture));
			pptInfosFacture.addCell(cell);
		}
		
		if (!dr.adresse3Dest.equals("")) {
			cell.setPhrase(new Phrase(dr.adresse3Dest, fontAdrFacture));
			pptInfosFacture.addCell(cell);
		}

		cell.setPhrase(new Phrase(dr.codePostalDest+" "+dr.villeDest, fontAdrFacture));
		pptInfosFacture.addCell(cell);

		if (!dr.paysDest.equals("")) {
			cell.setPhrase(new Phrase(dr.paysDest, fontAdrFacture));
			pptInfosFacture.addCell(cell);
		}

		cell.setPhrase(new Phrase("", fontTexte));
		pptInfosFacture.addCell(cell);
			
		return pptInfosFacture;
	}

	
	private PdfPTable makeLettre() throws DocumentException {

		PdfPTable pptLettre = new PdfPTable(2);
		int[] widths = {50,50};
		pptLettre.setWidths(widths);
		pptLettre.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(pptLettre.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(40);
		cell.setPaddingRight(40);

		cell.setPhrase(new Phrase("Objet : Lettre de remboursement", fontTexte));
		pptLettre.addCell(cell);

		cell.setPhrase(new Phrase("Date : "+ dr.dateCourante, fontTexte));
		pptLettre.addCell(cell);

		PdfPCell cellVide = new PdfPCell(pptLettre.getDefaultCell());
		cellVide.setBorder(PdfPCell.NO_BORDER);
		cellVide.setPhrase(new Phrase(""));
		cellVide.setFixedHeight(80);
		cellVide.setColspan(2);
		pptLettre.addCell(cellVide);
		
		cell.setColspan(2);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_JUSTIFIED);
		cell.setPhrase(new Phrase("Madame, Monsieur,", fontTexte));
		pptLettre.addCell(cell);
		cellVide.setFixedHeight(20);
		pptLettre.addCell(cellVide);
		cell.setPhrase(new Phrase("Par la pr\u00E9sente, nous avons le plaisir de vous confirmer le remboursement de "+ numFormatCap.format(dr.montant) +" \u20AC TTC, correspondant \u00E0 :", fontTexte));
		pptLettre.addCell(cell);
		pptLettre.addCell(cellVide);

		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		int nbAvoirs = dr.listeAvoirs.size();
		for (int i=0; i<nbAvoirs; i++) {
			cell.setPhrase(new Phrase("- AVOIR "+ dr.listeAvoirs.get(i), fontTexte));
			pptLettre.addCell(cell);
		}
		if (nbAvoirs>0) { pptLettre.addCell(cellVide); }
		int nbReglements = dr.listeReglements.size();
		for (int i=0; i<nbReglements; i++) {
			cell.setPhrase(new Phrase("- REGLEMENT "+ dr.listeReglements.get(i), fontTexte));
			pptLettre.addCell(cell);
		}
		if (nbReglements>0) { pptLettre.addCell(cellVide); }
		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_JUSTIFIED);
		if (!dr.commentaires.equals("")) {
			cell.setPhrase(new Phrase(dr.commentaires, fontTexte));
			pptLettre.addCell(cell);
			pptLettre.addCell(cellVide);
		}

		cell.setPhrase(new Phrase("En vous remerciant de votre confiance,", fontTexte));
		pptLettre.addCell(cell);
		pptLettre.addCell(cellVide);
		cell.setPhrase(new Phrase("Nous vous prions d'agr\u00E9er, Madame, Monsieur, l'assurance de notre consid\u00E9ration distingu\u00E9e.", fontTexte));
		pptLettre.addCell(cell);
		cellVide.setFixedHeight(30);
		pptLettre.addCell(cellVide);
		cell.setPhrase(new Phrase("La comptabilit\u00E9", fontTexte));
		pptLettre.addCell(cell);
		
		
		return pptLettre;
	}
	
	
	
	private PdfPTable makeInfosLegales() {
		
		PdfPTable pptLegales = new PdfPTable(1);
		pptLegales.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptLegales.getDefaultCell());		
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPhrase(new Phrase(dr.raisonSociale +" - "+ dr.typeSociete + (dr.capital>0?" au capital de "+ numFormatCap.format(dr.capital) +" Euros":"") +" - immatricul\u00E9e au RCS "+ dr.rcs +" "+ dr.siren +" - N\u00B0 TVA : "+ dr.tvaIntra, fontLegale));
		
		pptLegales.addCell(cell);
		
		return pptLegales;	
	}




} // fin GenPdfRemboursement
