/******************************************************************************/
/* OpenSi : Outils libres de gestion d'entreprise                             */
/* Copyright (C) 2003 Speedinfo.fr S.A.R.L.                                   */
/* Contact: contact@opensi.org                                                */
/*                                                                            */
/* This program is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU General Public License                */
/* as published by the Free Software Foundation; either version 2             */
/* of the License, or (at your option) any later version.                     */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/******************************************************************************/

/**
 * Gnration d'un journal des acomptes au format standard
 */

package org.opensi.facturation.actions.editions;


import java.util.Iterator;

import org.opensi.util.pdf.MultiPagesDocument;
import org.opensi.util.pdf.TableBuilder;
import org.opensi.util.tools.DateTime;
import org.opensi.util.tools.Formatter;

import com.lowagie.text.DocumentException;
import com.lowagie.text.Font;
import com.lowagie.text.Image;
import com.lowagie.text.Phrase;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;


public class PdfJournalAcomptes extends MultiPagesDocument {


private static final Font fontEnteteColonne = new Font(Font.HELVETICA, 10, Font.BOLD);
private static final Font fontLine = new Font(Font.HELVETICA , 9, Font.NORMAL);
private static final Font font9N = new Font(Font.HELVETICA, 9, Font.NORMAL);

private static final Font fontTitre2 = new Font(Font.HELVETICA, 10, Font.ITALIC);
private static final Font fontTotaux2 = new Font(Font.HELVETICA, 9, Font.NORMAL);
private static final Font fontEntete2 = new Font(Font.HELVETICA, 9, Font.BOLD);

private Formatter fnMonetaire = new Formatter("0.00", true);



private DataJournalAcomptes dja;	   // les donnes dynamiques du journal des acomptes

private int[] widthsCols;
private int nbCols;



  public PdfJournalAcomptes(String tempdir, DataJournalAcomptes dja) throws Exception {

		super(tempdir);
		
		this.dja = dja;		

		nbCols = 9;
		int[] widths = {9,9,22,10,10,10,10,10,10};
		widthsCols = widths;	
	}


	
	/* implmentation des fonctions de MultiPagesDocument */
	
	public PdfPTable firstHeader() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeEntete());
		tb.add(makeSpace(5));
		tb.add(makeEntetesColonnes());		
		
		return tb.getTable();	
	}
	
	
	public PdfPTable middleHeader(int lastLine) throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(firstHeader());
		
		return tb.getTable();
	}
	
	
	public PdfPTable lastFooter() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeFinPage());
		tb.add(makeSpace(5));
		tb.add(makeTotauxLast());
		
		return tb.getTable();
	}
	
	
	public PdfPTable middleFooter(int nbLinesPage, int lastLine) throws DocumentException {
			
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeFinPage());
		
		return tb.getTable();		
	}
	
	
	public PdfPTable bodyLine(int numLine) throws DocumentException {
	
		LigneJournal lj = dja.lignes.get(numLine);
		
		return makeLine(lj);
	}
	
	
	public PdfPTable emptyLine(float emptyHeight) throws DocumentException {
		
		PdfPTable pptSpace = new PdfPTable(nbCols);
		pptSpace.setWidthPercentage(100);
		pptSpace.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(emptyHeight);		
		cell.setPhrase(new Phrase(""));		
	
		for (int i=0; i<nbCols; i++) {
			pptSpace.addCell(cell);
		}
	
		return pptSpace;
	}
	
	
	public int nbLines() {
	
		return dja.getNbLignes();
	}
	
	
	public int nbEx() {
	
		return 1;
	}
	
	
	public Image imageFond() {
	
		return null;
	}



	/* mthodes internes */
		

	private PdfPTable makeSpace(int space) {
	
		PdfPTable pptSpace = new PdfPTable(1);
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setFixedHeight(space);
		cell.setPhrase(new Phrase(""));
	
		pptSpace.addCell(cell);
	
		return pptSpace;
	}
	

	private PdfPTable makeEntete() throws DocumentException {
	
		PdfPTable bandeau = new PdfPTable(1);

		bandeau.setWidthPercentage(100);

		PdfPTable entete = new PdfPTable(3);

		PdfPCell cellEntete = new PdfPCell(entete.getDefaultCell());

		cellEntete.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cellEntete.setBorder(PdfPCell.NO_BORDER);
		cellEntete.setMinimumHeight(10);

		cellEntete.setPhrase(new Phrase("Dossier : "+ dja.dossierId, font9N));
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase(dja.raisonSociale, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);
		cellEntete.setColspan(1);

		cellEntete.setPhrase(new Phrase("Edit\u00E9 le " + DateTime.formatTime(dja.dateCourante, "dd/MM/yyyy  HH:mm"), font9N));
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase(dja.adresse1, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);
		cellEntete.setColspan(1);

		cellEntete.setPhrase(new Phrase("", font9N));
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase(dja.adresse2, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);
		cellEntete.setColspan(1);

		cellEntete.setPhrase(new Phrase("Page : "+ currentPage +" / "+ totalPages, font9N));
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		entete.addCell(cellEntete);
		
		if (!dja.adresse3.equals("")) {
			cellEntete.setPhrase(new Phrase(dja.adresse3, font9N));
			cellEntete.setColspan(2);
			cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
			entete.addCell(cellEntete);
			cellEntete.setColspan(1);
			
			cellEntete.setPhrase(new Phrase("", font9N));
			cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			entete.addCell(cellEntete);
		}

		cellEntete.setPhrase(new Phrase(dja.codePostal +" "+ dja.ville, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase("Journal des acomptes en "+ dja.monnaie, new Font(Font.HELVETICA, 16, Font.BOLD)));
		cellEntete.setColspan(3);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase(""));
		cellEntete.setFixedHeight(10);
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase("P\u00E9riode du "+ DateTime.formatTime(dja.dateDebut, "dd/MM/yyyy") +" au "+ DateTime.formatTime(dja.dateFin, "dd/MM/yyyy"), font9N));
		cellEntete.setMinimumHeight(0);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		entete.addCell(cellEntete);

		PdfPCell cellBandeau = new PdfPCell(entete);
		cellBandeau.setPadding(10);

		bandeau.addCell(cellBandeau);
		
		return bandeau;		
	}


	private PdfPTable makeEntetesColonnes() throws DocumentException {
	
		PdfPTable pptCols = new PdfPTable(nbCols);
		pptCols.setWidthPercentage(100);
		pptCols.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptCols.getDefaultCell());		
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setPaddingBottom(4);
		cell.setMinimumHeight(40);
		
		cell.setPhrase(new Phrase("Date", fontEnteteColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("N\u00B0 Client", fontEnteteColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("Client", fontEnteteColonne));
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("N\u00B0 Compte", fontEnteteColonne));		
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("N\u00B0 Pice", fontEnteteColonne));
		pptCols.addCell(cell);			
		
		cell.setPhrase(new Phrase("Montant HT", fontEnteteColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("Montant TVA", fontEnteteColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("Montant TTC", fontEnteteColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("Solde TTC progressif", fontEnteteColonne));		
		pptCols.addCell(cell);
		
		return pptCols;
	}
	
	
	private PdfPTable makeFinPage() {
	
		PdfPTable pptFP = new PdfPTable(1);
		pptFP.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptFP.getDefaultCell());
		cell.setBorder(PdfPCell.TOP);
		cell.setPhrase(new Phrase(""));
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(1);
				
		pptFP.addCell(cell);
	
		return pptFP;
	}
	
	
	private PdfPTable makeLine(LigneJournal lj) throws DocumentException {
	
		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());

		cell.setBorder(PdfPCell.RIGHT + PdfPCell.LEFT);
		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		
		cell.setPhrase(new Phrase(DateTime.formatTime(lj.datePiece, "dd/MM/yyyy"), fontLine));
		pptLine.addCell(cell);
		
		cell.setPaddingLeft(5);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);

		cell.setPhrase(new Phrase(lj.numTiers, fontLine));
		pptLine.addCell(cell);

		cell.setPhrase(new Phrase(lj.tiers, fontLine));
		pptLine.addCell(cell);

		cell.setPhrase(new Phrase(lj.numCompte, fontLine));
		pptLine.addCell(cell);

		cell.setPaddingLeft(0);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		
		cell.setPhrase(new Phrase(lj.numPiece, fontLine));
		pptLine.addCell(cell);

		cell.setPaddingRight(3);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);		
		
		cell.setPhrase(new Phrase(fnMonetaire.get(lj.montantHT), fontLine));
		pptLine.addCell(cell);
		
		cell.setPhrase(new Phrase(fnMonetaire.get(lj.montantTVA), fontLine));
		pptLine.addCell(cell);
		
		cell.setPhrase(new Phrase(fnMonetaire.get(lj.montantTTC), fontLine));
		pptLine.addCell(cell);
		
		cell.setPhrase(new Phrase(fnMonetaire.get(lj.totalProgressif), fontLine));
		pptLine.addCell(cell);			

		return pptLine;	
	}
	
	
	private PdfPTable makeTotauxLast() throws DocumentException {		
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeCadreNetTVA());
		
		return tb.getTable();	
	}
	
	
	private PdfPTable makeCadreNetTVA() {
	
		PdfPTable pptTva = new PdfPTable(4);
		pptTva.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptTva.getDefaultCell());
	
		cell.setBorder(PdfPCell.BOX);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setColspan(4);
		cell.setPhrase(new Phrase("Ventilation nette globale    "+ dja.nbAcomptes +" acomptes", fontTitre2));
		pptTva.addCell(cell);
		cell.setColspan(1);
		cell.setPhrase(new Phrase("Taux TVA", fontEntete2));
		pptTva.addCell(cell);
		cell.setPhrase(new Phrase("Montant HT", fontEntete2));
		pptTva.addCell(cell);
		cell.setPhrase(new Phrase("Montant TVA", fontEntete2));
		pptTva.addCell(cell);
		cell.setPhrase(new Phrase("Montant TTC", fontEntete2));	
		pptTva.addCell(cell);
		
		Iterator<Double> li = dja.vtvasA.keySet().iterator();

		while (li.hasNext()) {
		
			VentilTVA vtva = dja.vtvasA.get(li.next());
		
			cell.setPaddingRight(3);
			cell.setBorder(PdfPCell.BOX-PdfPCell.TOP-PdfPCell.BOTTOM);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
			cell.setPhrase(new Phrase(fnMonetaire.get(vtva.tauxTVA) +" %"+ (vtva.npr?" NPR":""), fontTotaux2));
			pptTva.addCell(cell);
			cell.setPhrase(new Phrase(fnMonetaire.get(vtva.montantHT), fontTotaux2));		
			pptTva.addCell(cell);
			cell.setPhrase(new Phrase(fnMonetaire.get(vtva.montantTVA), fontTotaux2));
			pptTva.addCell(cell);
			cell.setPhrase(new Phrase(fnMonetaire.get(vtva.montantTTC), fontTotaux2));
			pptTva.addCell(cell);
		}		
		
		cell.setPaddingRight(3);
		cell.setBorder(PdfPCell.BOX);		
		cell.setPhrase(new Phrase("Totaux", fontEntete2));
		pptTva.addCell(cell);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPhrase(new Phrase(fnMonetaire.get(dja.montantAHT), fontTotaux2));		
		pptTva.addCell(cell);
		cell.setPhrase(new Phrase(fnMonetaire.get(dja.montantATVA), fontTotaux2));
		pptTva.addCell(cell);
		cell.setPhrase(new Phrase(fnMonetaire.get(dja.montantATTC), fontTotaux2));
		pptTva.addCell(cell);
		
		return pptTva;
	}
	

} // fin PdfJournalAcomptes
