/******************************************************************************/
/* OpenSi : Outils libres de gestion d'entreprise                             */
/* Copyright (C) 2003 Speedinfo.fr S.A.R.L.                                   */
/* Contact: contact@opensi.org                                                */
/*                                                                            */
/* This program is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU General Public License                */
/* as published by the Free Software Foundation; either version 2             */
/* of the License, or (at your option) any later version.                     */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/******************************************************************************/

/**
 * Gnration de l'tat des stocks en pdf
 */

package org.opensi.facturation.actions.editions;


import org.opensi.util.pdf.MultiPagesDocument;
import org.opensi.util.pdf.TableBuilder;
import org.opensi.util.tools.Formatter;

import com.lowagie.text.Chunk;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Font;
import com.lowagie.text.Image;
import com.lowagie.text.Phrase;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;



public class PdfEtatStock extends MultiPagesDocument {
	
	private static final Font font9N = new Font(Font.HELVETICA, 9, Font.NORMAL);
	private static final Font font9G = new Font(Font.HELVETICA, 9, Font.BOLD);
	private static final Font font9GI = new Font(Font.HELVETICA, 9, Font.BOLD + Font.ITALIC);
	private static final Font fontLigne = new Font(Font.HELVETICA, 8, Font.NORMAL);
	private static final Font fontColonne = new Font(Font.HELVETICA, 9, Font.BOLD);

	private DataEtatStock des;	   // les donnes dynamiques de l'tat des stocks
	
	Formatter fQte = new Formatter("0.###", true);
	Formatter fMon = new Formatter("0.00", true);

	private int[] widthsCols;
	private int nbCols;
	

	public PdfEtatStock(String tempdir, DataEtatStock des) throws Exception {
		
		super(tempdir);
		this.des = des;
		
		nbCols = 7;
		int widths[] = {17,37,10,9,9,9,9};
		widthsCols = widths;
	}
	
	
	
	/* implmentation des fonctions de MultiPagesDocument */
	
	public PdfPTable firstHeader() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeEntete());
		tb.add(makeSpace(10));
		tb.add(makeEntetesColonnes());
		tb.add(makeSpace(5));
		
		return tb.getTable();	
	}
	
	
	public PdfPTable middleHeader(int lastLine) throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(firstHeader());
		
		return tb.getTable();
	}
	
	
	public PdfPTable lastFooter() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeSpace(10));
		tb.add(makeTotaux());
		
		return tb.getTable();
	}
	
	
	public PdfPTable middleFooter(int nbLinesPage, int lastLine) throws DocumentException {
			
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeFinPage());
		
		return tb.getTable();		
	}
	
	
	public PdfPTable bodyLine(int numLine) throws DocumentException {
	
		LigneEtatStock les = des.lignes.get(numLine);
		
		return makeLine(les);
	}
	
	
	public PdfPTable emptyLine(float emptyHeight) throws DocumentException {
		
		PdfPTable pptSpace = new PdfPTable(nbCols);	
		return pptSpace;
	}
	
	
	public int nbLines() {
		
		return des.getNbLignes();
	}
	
	
	public int nbEx() {
	
		return 1;
	}
	
	
	public Image imageFond() {
	
		return null;
	}
	
	
	/* mthodes internes */

	private PdfPTable makeSpace(int space) throws DocumentException {

		PdfPTable pptSpace = new PdfPTable(1);
		pptSpace.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());

		cell.setPhrase(new Phrase(""));
		cell.setFixedHeight(space);
		cell.setBorder(PdfPCell.NO_BORDER);
		pptSpace.addCell(cell);

		return pptSpace;
	}
	
	
	private PdfPTable makeEntete() throws DocumentException {
		
		PdfPTable bandeau = new PdfPTable(1);
	
		bandeau.setWidthPercentage(100);
	
		PdfPTable entete = new PdfPTable(3);
	
		PdfPCell cellEntete = new PdfPCell(entete.getDefaultCell());
	
		cellEntete.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cellEntete.setBorder(PdfPCell.NO_BORDER);
		cellEntete.setMinimumHeight(10);
	
		cellEntete.setPhrase(new Phrase("Dossier : "+ des.dossierId, font9N));
		entete.addCell(cellEntete);
	
		cellEntete.setPhrase(new Phrase(des.raisonSociale, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);
		cellEntete.setColspan(1);
	
		cellEntete.setPhrase(new Phrase("Date : " + des.dateCourante, font9N));
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		entete.addCell(cellEntete);
	
		cellEntete.setPhrase(new Phrase(des.adresse1, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);
		cellEntete.setColspan(1);
	
		cellEntete.setPhrase(new Phrase("", font9N));
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		entete.addCell(cellEntete);
	
		cellEntete.setPhrase(new Phrase(des.adresse2, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);
		cellEntete.setColspan(1);
	
		cellEntete.setPhrase(new Phrase("Page : "+ currentPage +" / "+ totalPages, font9N));
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		entete.addCell(cellEntete);
		
		if (!des.adresse3.equals("")) {
			cellEntete.setPhrase(new Phrase(des.adresse3, font9N));
			cellEntete.setColspan(2);
			cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
			entete.addCell(cellEntete);
			cellEntete.setColspan(1);
			
			cellEntete.setPhrase(new Phrase("", font9N));
			cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			entete.addCell(cellEntete);
		}
	
		cellEntete.setPhrase(new Phrase(des.codePostal +" "+ des.ville, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);
	
		cellEntete.setPhrase(new Phrase("Etat des stocks"+ (des.existeDateArrete?" au "+ des.dateArrete:""), new Font(Font.HELVETICA, 16, Font.BOLD)));
		cellEntete.setColspan(3);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		entete.addCell(cellEntete);
	
		cellEntete.setPhrase(new Phrase(""));
		cellEntete.setFixedHeight(10);
		entete.addCell(cellEntete);	
	
		Phrase titreVal = new Phrase();
		Chunk ch1 = new Chunk("Valorisation en Co\u00FBt Unitaire Moyen Pond\u00E9r\u00E9", font9G);
		Chunk ch2 = new Chunk(" \"en fin de p\u00E9riode\"", font9GI);
		titreVal.add(ch1);
		titreVal.add(ch2);
	
		cellEntete.setMinimumHeight(0);
		cellEntete.setPhrase(titreVal);
		cellEntete.setMinimumHeight(0);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		entete.addCell(cellEntete);
	
		if (des.criteres.length()>0) {
			cellEntete.setPhrase(new Phrase(des.criteres, font9N));
			entete.addCell(cellEntete);
		}
	
		PdfPCell cellBandeau = new PdfPCell(entete);
		cellBandeau.setPadding(10);
	
		bandeau.addCell(cellBandeau);
		
		return bandeau;		
	}
	
	
	private PdfPTable makeEntetesColonnes() throws DocumentException {

		PdfPTable pptCols = new PdfPTable(nbCols);
		pptCols.setWidthPercentage(100);
		pptCols.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptCols.getDefaultCell());
		cell.setMinimumHeight(30);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);

		cell.setPhrase(new Phrase("R\u00E9f\u00E9rence", fontColonne));
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("D\u00E9signation", fontColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("Marque", fontColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("Valeur totale", fontColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("Valeur unitaire", fontColonne));
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("Stock r\u00E9el", fontColonne));
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("Stock disponible", fontColonne));
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("Stock virtuel", fontColonne));
		pptCols.addCell(cell);

		return pptCols;
	}
	
	
	private PdfPTable makeFinPage() {
		
		PdfPTable pptFP = new PdfPTable(1);
		pptFP.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptFP.getDefaultCell());
		cell.setBorder(PdfPCell.TOP);
		cell.setPhrase(new Phrase(""));
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(1);
				
		pptFP.addCell(cell);
	
		return pptFP;
	}

	
	private PdfPTable makeTotaux() throws DocumentException {
		
		PdfPTable pptTotaux = new PdfPTable(3);
		int widths[] = {54,28,18};
		pptTotaux.setWidthPercentage(100);
		pptTotaux.setWidths(widths);

		PdfPCell cell = new PdfPCell(pptTotaux.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setPaddingRight(4);

		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPhrase(new Phrase("", font9G));
		pptTotaux.addCell(cell);
		cell.setBorder(PdfPCell.BOX);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase("Valorisation totale HT stock r\u00E9el :", font9G));
		pptTotaux.addCell(cell);	
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);		
		cell.setPhrase(new Phrase(fMon.get(des.valorisationTotale), font9G));
		pptTotaux.addCell(cell);

		return pptTotaux;
	}


	private PdfPTable makeLine(LigneEtatStock les) throws DocumentException {

		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setPhrase(new Phrase(les.numero, fontLigne));
		pptLine.addCell(cell);

		cell.setPhrase(new Phrase(les.designation, fontLigne));
		pptLine.addCell(cell);

		cell.setPhrase(new Phrase(les.marque, fontLigne));
		pptLine.addCell(cell);

		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPaddingRight(4);
		
		cell.setPhrase(new Phrase(fMon.get(les.valorisation * les.reel), fontLigne));
		pptLine.addCell(cell);
		
		cell.setPhrase(new Phrase(fMon.get(les.valorisation), fontLigne));
		pptLine.addCell(cell);

		cell.setPhrase(new Phrase(fQte.get(les.reel), fontLigne));
		pptLine.addCell(cell);

		cell.setPhrase(new Phrase(fQte.get(les.dispo), fontLigne));
		pptLine.addCell(cell);

		cell.setPhrase(new Phrase(fQte.get(les.virtuel), fontLigne));
		pptLine.addCell(cell);

		return pptLine;
	}


} // fin PdfEtatStock
