/******************************************************************************/
/* OpenSi : Outils libres de gestion d'entreprise                             */
/* Copyright (C) 2003 Speedinfo.fr S.A.R.L.                                   */
/* Contact: contact@opensi.org                                                */
/*                                                                            */
/* This program is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU General Public License                */
/* as published by the Free Software Foundation; either version 2             */
/* of the License, or (at your option) any later version.                     */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/******************************************************************************/

/**
 * Gnration d'un avoir au format standard
 */

package org.opensi.facturation.actions.documents.modeles;


import java.awt.Color;
import java.io.IOException;
import java.sql.Date;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.SimpleDateFormat;
import java.util.Iterator;

import org.opensi.util.calcul.CalculVentilTVA;
import org.opensi.util.pdf.PdfWidths;
import org.opensi.util.pdf.TableBuilder;

import com.lowagie.text.DocumentException;
import com.lowagie.text.Font;
import com.lowagie.text.Image;
import com.lowagie.text.Phrase;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;


public class AvoirStd extends AvoirPDF {


private static final Font fontAdrGras = new Font(Font.HELVETICA, 11, Font.BOLD);
private static final Font fontAdrSociete = new Font(Font.HELVETICA, 10, Font.NORMAL);
private static final Font fontAdrSocieteGras = new Font(Font.HELVETICA, 11, Font.BOLD);
private static final Font fontAdresse = new Font(Font.HELVETICA, 8, Font.NORMAL);
private static final Font fontAdrContact = new Font(Font.HELVETICA, 7, Font.NORMAL);
private static final Font fontTitre = new Font(Font.HELVETICA , 16, Font.BOLD, Color.WHITE);
private static final Font fontWeb = new Font(Font.HELVETICA, 10, Font.ITALIC);
private static final Font fontCadre1 = new Font(Font.HELVETICA, 10, Font.BOLD);
private static final Font fontEnteteColonne = new Font(Font.HELVETICA, 9, Font.BOLD);
private static final Font fontInfosFact = new Font(Font.HELVETICA , 10, Font.NORMAL);
private static final Font fontLegale = new Font(Font.HELVETICA , 7, Font.NORMAL);
private static final Font fontLine = new Font(Font.HELVETICA , 8, Font.NORMAL);
private static final Font fontModalites = new Font(Font.HELVETICA , 10, Font.BOLD);
private static final Font fontTotaux = new Font(Font.HELVETICA , 10, Font.NORMAL);
private static final Font fontTTC = new Font(Font.HELVETICA , 10, Font.BOLD);
private static final Font fontArtNom = new Font(Font.HELVETICA, 9, Font.ITALIC);
private static final Font fontSmallTitre = new Font(Font.HELVETICA , 8, Font.BOLD+Font.ITALIC, Color.WHITE);


private DecimalFormat numFormat = new DecimalFormat("0.00");
private DecimalFormat numFormat4 = new DecimalFormat("0.00##");
private DecimalFormat numFormatCap = new DecimalFormat("0.##");
private SimpleDateFormat formatDate = new SimpleDateFormat("dd/MM/yyyy");


private float[] widthsCols;
private int nbCols;

private boolean rist;
private boolean coul;
private boolean tail;
private boolean unite;
private boolean nbPieces;
private boolean numLot;
private boolean datePeremption;

private double sousTotalCourant = 0;

private Image logo;
private boolean afficherAdrFact;


	public AvoirStd(String tempdir, DataAvoir id) throws Exception {

		super(tempdir,id);
		
		DecimalFormatSymbols dfs = new DecimalFormatSymbols();
		dfs.setDecimalSeparator('.');
		dfs.setGroupingSeparator(' ');
		
		numFormat.setDecimalFormatSymbols(dfs);
		numFormat.setNegativeSuffix("");
		numFormat.setNegativePrefix("-");
		numFormat.setPositiveSuffix(" ");
		numFormat.setGroupingUsed(true);
		numFormat.setGroupingSize(3);		
		
		numFormat4.setDecimalFormatSymbols(dfs);
		numFormat4.setGroupingUsed(true);
		numFormat4.setGroupingSize(3);
		
		numFormatCap.setDecimalFormatSymbols(dfs);
		numFormatCap.setGroupingUsed(true);
		numFormatCap.setGroupingSize(3);		
		
		try {
			logo = Image.getInstance(id.urlLogo);
			if (logo.getDpiX()!=0 && logo.getDpiY()!=0)
      	logo.scaleAbsolute(logo.getPlainWidth() * 72 / logo.getDpiX(), logo.getPlainHeight() * 72 / logo.getDpiY());
		}
		catch (IOException e) {
			logo = null;
		}

		this.afficherAdrFact = (!id.raisonSocialeDest.equals(id.denominationEnvoi) || !id.adresseDest.equals(id.adresse1Envoi) || !id.compAdresseDest.equals(id.adresse2Envoi) || !id.adresseDest3.equals(id.adresse3Envoi)
			|| !id.codePostalDest.equals(id.codePostalEnvoi) || !id.villeDest.equals(id.villeEnvoi) || !id.paysDest.equals(id.paysEnvoi));

		rist = id.rist;
		tail = id.tail;
		coul = id.coul;
		unite = id.unite;
		nbPieces = id.nbPieces;
		numLot = id.numLot;
		datePeremption = id.datePeremption;
		
		PdfWidths pw = new PdfWidths();
		
		pw.addColonneVariableMinMax(2, 12, 20);
		pw.addColonneVariableMin(4,20);
		if (coul) pw.addColonneVariableMin(1, 6);
		if (tail) pw.addColonneVariableMin(1, 6);
		if (numLot) pw.addColonneVariableMinMax(2, 6, 10);
		if (nbPieces) pw.addColonneFixe(5);
		if (datePeremption) pw.addColonneFixe(8);
		pw.addColonneVariableMin(1, 8);
		if (unite) pw.addColonneFixe(5);
		pw.addColonneVariableMin(1, 6);
		if (rist) pw.addColonneFixe(6);
		pw.addColonneVariableMin(1, 9);
		
		nbCols = pw.getNbCols();
		widthsCols = pw.getWidthsCols();
	
	}


	
	/* implmentation des fonctions de MultiPagesDocument */
	
	public PdfPTable firstHeader() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeEntete());
		tb.add(makeSpace(5));
		tb.add(makeTitre());
		tb.add(makeSpace(5));
		if (this.afficherAdrFact || id.imprimerAdrLiv) {
			tb.add(makeAdresses());
			tb.add(makeSpace(5));
		}
		if (id.codeFournisseur.length()>0) {
			tb.add(makeResponsable());
			tb.add(makeSpace(5));
		}
		tb.add(makeEntetesColonnes());
		
		return tb.getTable();	
	}
	
	
	public PdfPTable middleHeader(int lastLine) throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeEntete());
		tb.add(makeSpace(5));
		tb.add(makeTitre());
		tb.add(makeSpace(5));
		tb.add(makeEntetesColonnes());
		
		tb.add(makeSousTotal(id.calcSousTotal(lastLine)));
		
		return tb.getTable();
	}
	
	
	public PdfPTable lastFooter() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		if (id.comFin.length()>0 || id.mentions.length()>0) {
			tb.add(makeZoneCommentaire());
			tb.add(emptyLine(10));
		}
		tb.add(makeTotauxLast());
		tb.add(makeSpace(3));
		tb.add(makeInfosLegales());
		
		return tb.getTable();
	}	
	
	
	public PdfPTable middleFooter(int nbLinesPage, int lastLine) throws DocumentException {
	
		sousTotalCourant = id.calcSousTotal(lastLine);
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(emptyLine(10));
		tb.add(makeTotauxMiddle());
		tb.add(makeSpace(3));
		tb.add(makeInfosLegales());		
		
		return tb.getTable();		
	}
	
	
	public PdfPTable bodyLine(int numLine) throws DocumentException {
	
		LigneAvoir la = id.lignes.get(numLine);
	
		switch (la.type) {			
			case 'A':	return makeLineArticle(la);
			case 'N': return makeLineNomenclature(la);
			case 'C':
			default: return makeLineCommentaire(la);
		}		
	}
	
	
	public PdfPTable emptyLine(float emptyHeight) throws DocumentException {
		
		PdfPTable pptSpace = new PdfPTable(nbCols);
		pptSpace.setWidthPercentage(100);
		pptSpace.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(emptyHeight);		
		cell.setPhrase(new Phrase(""));		
	
		for (int i=0; i<nbCols; i++) {
			pptSpace.addCell(cell);
		}
	
		return pptSpace;
	}
	
	
	public int nbLines() {
	
		return id.getNbLignes();
	}
	
	
	public int nbEx() {
	
		return (multi?id.getNbEx():1);
	}
	
	
	public Image imageFond() {
		if (id.apercu) {	
			try {
				Image fond = Image.getInstance(id.urlApercu);
				fond.scaleAbsolute(324, 468);
				fond.setAbsolutePosition(135,100);
				
				return fond;
			}
			catch (Exception e) {
				return null;
			}
		}
		else
			return null;
	}



	/* mthodes internes */

	private PdfPTable makeSpace(int space) {
	
		PdfPTable pptSpace = new PdfPTable(1);
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setFixedHeight(space);
		cell.setPhrase(new Phrase(""));
	
		pptSpace.addCell(cell);
	
		return pptSpace;
	}
	
	
	private PdfPTable makeEntete() throws DocumentException {
		
		PdfPTable pptEntete = new PdfPTable(2);
		pptEntete.setWidthPercentage(100);
		int[] widths = {50,50};
		pptEntete.setWidths(widths);
		
		PdfPTable pptEnteteGauche = new PdfPTable(1);
		pptEnteteGauche.setWidthPercentage(100);
		PdfPTable pptEnteteDroite = new PdfPTable(1);
		pptEnteteDroite.setWidthPercentage(100);
		
		PdfPCell cellSociete = new PdfPCell(makeInfosSociete());
		cellSociete.setMinimumHeight(115);
		cellSociete.setBorder(PdfPCell.NO_BORDER);
		pptEnteteGauche.addCell(cellSociete);
		PdfPCell cellVide = new PdfPCell(pptEnteteGauche.getDefaultCell());
		cellVide.setBorder(PdfPCell.NO_BORDER);
		cellVide.setPhrase(new Phrase(""));
		pptEnteteGauche.addCell(cellVide);
		
		PdfPCell cellFacture = new PdfPCell(makeInfosFacture());
		cellFacture.setFixedHeight(115);
		cellFacture.setBorder(PdfPCell.NO_BORDER);
		pptEnteteDroite.addCell(cellFacture);		
		PdfPCell cellAdrEnvoi = new PdfPCell(makeAdrEnvoi());
		cellAdrEnvoi.setMinimumHeight(80);
		cellAdrEnvoi.setBorder(PdfPCell.NO_BORDER);
		pptEnteteDroite.addCell(cellAdrEnvoi);
		pptEnteteDroite.addCell(cellVide);
		
		PdfPCell cellPartieGauche = new PdfPCell(pptEnteteGauche);
		cellPartieGauche.setBorder(PdfPCell.NO_BORDER);
		PdfPCell cellPartieDroite = new PdfPCell(pptEnteteDroite);		
		cellPartieDroite.setBorder(PdfPCell.NO_BORDER);
		pptEntete.addCell(cellPartieGauche);
		pptEntete.addCell(cellPartieDroite);
		
		return pptEntete;		
	}
	
	
	private PdfPTable makeAdresses() throws DocumentException {
		
		PdfPTable pptAdresses = new PdfPTable(!this.afficherAdrFact || !id.imprimerAdrLiv?1:3);
		pptAdresses.setWidthPercentage(100);
		if (this.afficherAdrFact && id.imprimerAdrLiv) {
			int[] widths = {49,2,49};
			pptAdresses.setWidths(widths);
		}
		
		PdfPTable pptAdresseGauche = new PdfPTable(1);
		pptAdresseGauche.setWidthPercentage(100);
		PdfPTable pptAdresseDroite = new PdfPTable(1);
		pptAdresseDroite.setWidthPercentage(100);
		
		if (this.afficherAdrFact) {
			PdfPCell cellAdrFact = new PdfPCell(makeAdrFact());
			cellAdrFact.setBorder(PdfPCell.NO_BORDER);
			pptAdresseGauche.addCell(cellAdrFact);
		}
		if (id.imprimerAdrLiv) {
			PdfPCell cellAdrLiv = new PdfPCell(makeAdrLiv());
			cellAdrLiv.setBorder(PdfPCell.NO_BORDER);
			pptAdresseDroite.addCell(cellAdrLiv);
		}
		
		PdfPCell cellPartieGauche = new PdfPCell(pptAdresseGauche);
		cellPartieGauche.setBorder(PdfPCell.NO_BORDER);
		PdfPCell cellVide = new PdfPCell(pptAdresses.getDefaultCell());
		cellVide.setBorder(PdfPCell.NO_BORDER);
		cellVide.setPhrase(new Phrase(""));
		PdfPCell cellPartieDroite = new PdfPCell(pptAdresseDroite);		
		cellPartieDroite.setBorder(PdfPCell.NO_BORDER);
		if (this.afficherAdrFact) { pptAdresses.addCell(cellPartieGauche); }
		if (this.afficherAdrFact && id.imprimerAdrLiv) { pptAdresses.addCell(cellVide); }
		if (id.imprimerAdrLiv) { pptAdresses.addCell(cellPartieDroite); }
		
		return pptAdresses;
	}


	private PdfPTable makeInfosSociete() {
	
		PdfPTable pptInfos = new PdfPTable(1);
		
		PdfPCell cell;
				
		if (logo != null) {
			cell = new PdfPCell(logo, false);

			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

			pptInfos.addCell(cell);

			cell.setImage(null);

			cell.setMinimumHeight(5);
			cell.setPhrase(new Phrase("", fontAdrSociete));
			pptInfos.addCell(cell);
			cell.setMinimumHeight(0);
		}
		else {
			cell = new PdfPCell(pptInfos.getDefaultCell());
			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		}	
		
		if (!id.logoAdr) {
			cell.setPhrase(new Phrase(id.raisonSociale, fontAdrSocieteGras));		
			pptInfos.addCell(cell);
			cell.setPhrase(new Phrase(id.adresse1, fontAdrSociete));		
			pptInfos.addCell(cell);
			if (!id.adresse2.equals("")) {
				cell.setPhrase(new Phrase(id.adresse2, fontAdrSociete));
				pptInfos.addCell(cell);
			}
			if (!id.adresse3.equals("")) {
				cell.setPhrase(new Phrase(id.adresse3, fontAdrSociete));
				pptInfos.addCell(cell);
			}
			cell.setPhrase(new Phrase(id.codePostal+" "+id.ville, fontAdrSociete));		
			pptInfos.addCell(cell);

			String telfax = "";

			if (id.tel.length()>0) {
				telfax = "T\u00E9l: "+ id.tel;
				if (id.fax.length()>0) {
					telfax += " - Fax: "+ id.fax;
				}
			}
			else if (id.fax.length()>0) {
				telfax = "Fax: "+ id.fax;
			}		

			if (telfax.length()>0) {
				cell.setPhrase(new Phrase(telfax, fontAdrSociete));		
				pptInfos.addCell(cell);
			}

			if (!id.email.equals("")) {
				cell.setPhrase(new Phrase("e-mail: "+ id.email, fontAdrSociete));		
				pptInfos.addCell(cell);
			}
			if (!id.web.equals("")) {
				cell.setPhrase(new Phrase(id.web, fontWeb));		
				pptInfos.addCell(cell);
			}
		}
		
		cell.setPhrase(new Phrase(""));		
		pptInfos.addCell(cell);
		
		return pptInfos;	
	}
	
	
	private PdfPTable makeAdrFact() throws DocumentException {
		
		PdfPTable pptAdresses = new PdfPTable(id.interlocuteur.equals("")?1:2);
		pptAdresses.setWidthPercentage(100);
		if (!id.interlocuteur.equals("")) {
			int[] widths = {50,50};
			pptAdresses.setWidths(widths);
		}
		
		PdfPCell cell = new PdfPCell(pptAdresses.getDefaultCell());
		if (!id.interlocuteur.equals("")) { cell.setColspan(2); }
		cell.setBackgroundColor(new Color(175,175,175));
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
		cell.setPhrase(new Phrase("FACTURATION", fontSmallTitre));
		pptAdresses.addCell(cell);
		
		PdfPTable pptAdresseGauche = new PdfPTable(1);
		pptAdresseGauche.setWidthPercentage(100);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setBackgroundColor(Color.WHITE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(id.raisonSocialeDest, fontAdresse));
		pptAdresseGauche.addCell(cell);
		cell.setPhrase(new Phrase(id.adresseDest, fontAdresse));
		pptAdresseGauche.addCell(cell);
		if (!id.compAdresseDest.equals("")) {
			cell.setPhrase(new Phrase(id.compAdresseDest, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		if (!id.adresseDest3.equals("")) {
			cell.setPhrase(new Phrase(id.adresseDest3, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		cell.setPhrase(new Phrase(id.codePostalDest+" "+id.villeDest, fontAdresse));
		pptAdresseGauche.addCell(cell);
		if (!id.paysDest.equals("")) {
			cell.setPhrase(new Phrase(id.paysDest, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		
		PdfPCell cellPartieGauche = new PdfPCell(pptAdresseGauche);
		pptAdresses.addCell(cellPartieGauche);
		
		if (!id.interlocuteur.equals("")) {
			PdfPTable pptAdresseDroite = new PdfPTable(1);
			pptAdresseDroite.setWidthPercentage(100);
			
			cell.setPhrase(new Phrase("Contact : "+id.interlocuteur, fontAdrContact));
			pptAdresseDroite.addCell(cell);
			if (!id.interMail.equals("")) {
				cell.setPhrase(new Phrase("E-Mail : "+id.interMail, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			if (!id.interTel.equals("")) {
				cell.setPhrase(new Phrase("T\u00E9l : "+id.interTel, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			if (!id.interFax.equals("")) {
				cell.setPhrase(new Phrase("Fax : "+id.interFax, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			PdfPCell cellPartieDroite = new PdfPCell(pptAdresseDroite);
			pptAdresses.addCell(cellPartieDroite);
		}
		
		return pptAdresses;
	}
	
	
	private PdfPTable makeAdrLiv() throws DocumentException {
		
		PdfPTable pptAdresses = new PdfPTable(id.interlocuteurLiv.equals("")?1:2);
		pptAdresses.setWidthPercentage(100);
		if (!id.interlocuteurLiv.equals("")) {
			int[] widths = {50,50};
			pptAdresses.setWidths(widths);
		}
		
		PdfPCell cell = new PdfPCell(pptAdresses.getDefaultCell());
		if (!id.interlocuteurLiv.equals("")) { cell.setColspan(2); }
		cell.setBackgroundColor(new Color(175,175,175));
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
		cell.setPhrase(new Phrase("LIVRAISON", fontSmallTitre));
		pptAdresses.addCell(cell);
		
		PdfPTable pptAdresseGauche = new PdfPTable(1);
		pptAdresseGauche.setWidthPercentage(100);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setBackgroundColor(Color.WHITE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(id.denominationLiv, fontAdresse));
		pptAdresseGauche.addCell(cell);
		cell.setPhrase(new Phrase(id.adresse1Liv, fontAdresse));
		pptAdresseGauche.addCell(cell);
		if (!id.adresse2Liv.equals("")) {
			cell.setPhrase(new Phrase(id.adresse2Liv, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		if (!id.adresse3Liv.equals("")) {
			cell.setPhrase(new Phrase(id.adresse3Liv, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		cell.setPhrase(new Phrase(id.codePostalLiv+" "+id.villeLiv, fontAdresse));
		pptAdresseGauche.addCell(cell);
		if (!id.paysLiv.equals("")) {
			cell.setPhrase(new Phrase(id.paysLiv, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		
		PdfPCell cellPartieGauche = new PdfPCell(pptAdresseGauche);
		pptAdresses.addCell(cellPartieGauche);
		
		if (!id.interlocuteurLiv.equals("")) {
			PdfPTable pptAdresseDroite = new PdfPTable(1);
			pptAdresseDroite.setWidthPercentage(100);
			
			cell.setPhrase(new Phrase("Contact : "+id.interlocuteurLiv, fontAdrContact));
			pptAdresseDroite.addCell(cell);
			if (!id.interMailLiv.equals("")) {
				cell.setPhrase(new Phrase("E-Mail : "+id.interMailLiv, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			if (!id.interTelLiv.equals("")) {
				cell.setPhrase(new Phrase("T\u00E9l : "+id.interTelLiv, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			if (!id.interFaxLiv.equals("")) {
				cell.setPhrase(new Phrase("Fax : "+id.interFaxLiv, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			PdfPCell cellPartieDroite = new PdfPCell(pptAdresseDroite);
			pptAdresses.addCell(cellPartieDroite);
		}
		
		return pptAdresses;
	}
	
	
	private PdfPTable makeAdrEnvoi() throws DocumentException {

		PdfPTable pptInfosFacture = new PdfPTable(2);

		PdfPCell cell = new PdfPCell(pptInfosFacture.getDefaultCell());

		cell.setColspan(2);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setMinimumHeight(0);
		cell.setPaddingLeft(0);
		cell.setPaddingBottom(0);
		
		cell.setBorder(PdfPCell.NO_BORDER);
		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(id.denominationEnvoi, fontAdrGras));
		pptInfosFacture.addCell(cell);
		
		cell.setPhrase(new Phrase(id.adresse1Envoi, fontAdrGras));
		pptInfosFacture.addCell(cell);
		
		if (!id.adresse2Envoi.equals("")) {
			cell.setPhrase(new Phrase(id.adresse2Envoi, fontAdrGras));
			pptInfosFacture.addCell(cell);
		}
		
		if (!id.adresse3Envoi.equals("")) {
			cell.setPhrase(new Phrase(id.adresse3Envoi, fontAdrGras));
			pptInfosFacture.addCell(cell);
		}
		
		cell.setPhrase(new Phrase(id.codePostalEnvoi+" "+id.villeEnvoi, fontAdrGras));
		pptInfosFacture.addCell(cell);
		
		if (!id.paysEnvoi.equals("")) {
			cell.setPhrase(new Phrase(id.paysEnvoi, fontAdrGras));
			pptInfosFacture.addCell(cell);
		}
		
		cell.setPhrase(new Phrase("", fontLine));
		pptInfosFacture.addCell(cell);

		PdfPTable pptGlobale = new PdfPTable(2);
		int[] widths = {85,15};
		pptGlobale.setWidths(widths);

		PdfPCell cellGlob = new PdfPCell(pptInfosFacture);
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		cellGlob = new PdfPCell(pptGlobale.getDefaultCell());
		cellGlob.setPhrase(new Phrase(""));
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		return pptGlobale;
	}


	private PdfPTable makeInfosFacture() throws DocumentException {
		
		PdfPTable pptInfosFacture = new PdfPTable(2);
		
		int[] widths = {60,40};
		pptInfosFacture.setWidths(widths);
		
		PdfPCell cell = new PdfPCell(pptInfosFacture.getDefaultCell());
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setPaddingLeft(4);
		cell.setPaddingBottom(4);
		
		cell.setPhrase(new Phrase("Avoir N\u00B0 "+ id.numeroAvoir, fontInfosFact));
		pptInfosFacture.addCell(cell);
		
		cell.setPhrase(new Phrase("Date : "+ formatDate.format(new Date(id.dateAvoir)), fontInfosFact));
		pptInfosFacture.addCell(cell);
		
		String codeClient = (id.clientId.length()>0?"Client N\u00B0 "+ id.clientId +" - ":"Client : ")+ id.raisonSocialeDest;
		
		cell.setColspan(2);
		cell.setPhrase(new Phrase(codeClient, fontInfosFact));
		pptInfosFacture.addCell(cell);

		if (id.interlocuteur.length()>0) {
			cell.setPhrase(new Phrase("Interlocuteur : "+ id.interlocuteur, fontInfosFact));
			pptInfosFacture.addCell(cell);
		}
		
		cell.setBorder(PdfPCell.NO_BORDER);
		
		cell.setPhrase(new Phrase("", fontLine));
		cell.setFixedHeight(5);
		pptInfosFacture.addCell(cell);
		cell.setMinimumHeight(0);
		
		cell.setPhrase(new Phrase("Page "+ currentPage +" / "+ totalPages, fontInfosFact));
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		pptInfosFacture.addCell(cell);
		
		cell.setPhrase(new Phrase("", fontLine));
		cell.setFixedHeight(55);
		pptInfosFacture.addCell(cell);
		
		return pptInfosFacture;		
	}


	private PdfPTable makeTitre() {
	
		PdfPTable pptTitre = new PdfPTable(1);
		pptTitre.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptTitre.getDefaultCell());		
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setBackgroundColor(new Color(175,175,175));
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setFixedHeight(30);
		cell.setPaddingBottom(5);
		cell.setPhrase(new Phrase("A     V     O     I     R", fontTitre));
		
		pptTitre.addCell(cell);
		
		return pptTitre;	
	}
	

	private PdfPTable makeResponsable() {
	
		PdfPTable pptResponsable = new PdfPTable(1);
		pptResponsable.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptResponsable.getDefaultCell());		
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setFixedHeight(20);
		cell.setPaddingLeft(5);
		
		cell.setPhrase(new Phrase("Num\u00E9ro Fournisseur : "+ id.codeFournisseur, fontCadre1));
		pptResponsable.addCell(cell);
		
		return pptResponsable;
	}
	
	
	private PdfPTable makeEntetesColonnes() throws DocumentException {
	
		PdfPTable pptCols = new PdfPTable(nbCols);
		pptCols.setWidthPercentage(100);
		pptCols.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptCols.getDefaultCell());		
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setPaddingBottom(4);
		cell.setMinimumHeight(40);
		
		cell.setPhrase(new Phrase("R\u00E9f\u00E9rence", fontEnteteColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("D\u00E9signation", fontEnteteColonne));
		pptCols.addCell(cell);
		
		if (coul) {
			cell.setPhrase(new Phrase("Coul.", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		
		if (tail) {
			cell.setPhrase(new Phrase("Taille", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		
		if (numLot) {
			cell.setPhrase(new Phrase("N\u00B0 Lot", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		if (nbPieces) {
			cell.setPhrase(new Phrase("Nb Pc", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		if (datePeremption) {
			cell.setPhrase(new Phrase("DLC", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		
		cell.setPhrase(new Phrase("Quantit\u00E9", fontEnteteColonne));
		pptCols.addCell(cell);
		
		if (unite) {
			cell.setPhrase(new Phrase("Unit\u00E9", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		
		cell.setPhrase(new Phrase((id.editionTTC?"PU TTC":"PU HT"), fontEnteteColonne));
		pptCols.addCell(cell);
		if (rist) {
			cell.setPhrase(new Phrase("Rist. %", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		cell.setPhrase(new Phrase((id.editionTTC?"Montant TTC":"Montant HT"), fontEnteteColonne));		
		pptCols.addCell(cell);
		
		return pptCols;
	}	
	
	
	private PdfPTable makeLineArticle(LigneAvoir la) throws DocumentException {
	
		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);

		cell.setPaddingLeft(5);
		cell.setPhrase(new Phrase(la.reference, fontLine));
		pptLine.addCell(cell);		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(la.designation, fontLine));
		pptLine.addCell(cell);

		if (coul) {
			cell.setPhrase(new Phrase(la.couleur, fontLine));
			pptLine.addCell(cell);
		}
		if (tail) {
			cell.setPhrase(new Phrase(la.taille, fontLine));
			pptLine.addCell(cell);
		}
		
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		
		if (numLot) {
			cell.setPhrase(new Phrase((la.num_lot), fontLine));
			pptLine.addCell(cell);
		}
		if (nbPieces) {
			if (la.nb_pieces != 0)
				cell.setPhrase(new Phrase(""+numFormatCap.format(la.nb_pieces), fontLine));
			else
				cell.setPhrase(new Phrase("", fontLine));			
			pptLine.addCell(cell);
		}
		if (datePeremption) {
			if (la.date_peremption != 0)
				cell.setPhrase(new Phrase(""+ formatDate.format(new Date(la.date_peremption)), fontLine));
			else
				cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
		}

		cell.setPhrase(new Phrase(""+ numFormatCap.format(la.quantite), fontLine));
		pptLine.addCell(cell);
		if (unite) {
			cell.setPhrase(new Phrase(la.unite, fontLine));
			pptLine.addCell(cell);
		}
		
		cell.setPaddingRight(3);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPhrase(new Phrase(numFormat4.format(la.prixUnitaire), fontLine));
		pptLine.addCell(cell);
		
		if (rist) {
			if (la.ristourne!=0) {
				cell.setPhrase(new Phrase(numFormat.format(la.ristourne), fontLine));
			}
			else {
				cell.setPhrase(new Phrase("", fontLine));
			}
			pptLine.addCell(cell);
		}

		cell.setPhrase(new Phrase(numFormat.format(la.montant), fontLine));		
		pptLine.addCell(cell);

		if (la.detail_1.length()>0) {
			cell.setPaddingLeft(5);
			cell.setPaddingRight(0);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase(la.detail_1, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			if (rist) pptLine.addCell(cell);
			if (coul) pptLine.addCell(cell);		
			if (tail) pptLine.addCell(cell);
			if (unite) pptLine.addCell(cell);
			if (numLot) pptLine.addCell(cell);
			if (nbPieces) pptLine.addCell(cell);
			if (datePeremption) pptLine.addCell(cell);
		}

		if (la.detail_2.length()>0) {
			cell.setPaddingLeft(5);
			cell.setPaddingRight(0);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase(la.detail_2, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			if (rist) pptLine.addCell(cell);
			if (coul) pptLine.addCell(cell);		
			if (tail) pptLine.addCell(cell);
			if (unite) pptLine.addCell(cell);
			if (numLot) pptLine.addCell(cell);
			if (nbPieces) pptLine.addCell(cell);
			if (datePeremption) pptLine.addCell(cell);
		}

		return pptLine;	
	}
		
	
	private PdfPTable makeLineNomenclature(LigneAvoir la) throws DocumentException {
	
		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());		
		
		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		
		PdfPTable pptArtNom = new PdfPTable(unite?3:2);
		pptArtNom.setWidthPercentage(100);
		int[] widths = {10,90};
		int[] widthsUnite = {10,10,80};
		pptArtNom.setWidths(unite?widthsUnite:widths);
		
		PdfPCell cellArtNom = new PdfPCell(pptArtNom.getDefaultCell());
		cellArtNom.setBorder(PdfPCell.NO_BORDER);
		
		cellArtNom.setPhrase(new Phrase(numFormatCap.format(la.quantite), fontArtNom));
		pptArtNom.addCell(cellArtNom);
		
		if (unite) {
			cellArtNom.setPhrase(new Phrase(la.unite, fontArtNom));
			pptArtNom.addCell(cellArtNom);
		}
		
		cellArtNom.setPhrase(new Phrase(la.designation, fontArtNom));
		pptArtNom.addCell(cellArtNom);
		
		PdfPCell cellDesc = new PdfPCell(pptArtNom);
		cellDesc.setBorder(PdfPCell.NO_BORDER);
		cellDesc.setPaddingLeft(10);
		pptLine.addCell(cellDesc);
		
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		
		if (rist) pptLine.addCell(cell);		
		if (coul) pptLine.addCell(cell);		
		if (tail) pptLine.addCell(cell);
		if (unite) pptLine.addCell(cell);
		if (numLot) pptLine.addCell(cell);
		if (nbPieces) pptLine.addCell(cell);
		if (datePeremption) pptLine.addCell(cell);
		
		return pptLine;	
	}
	
		
	private PdfPTable makeLineCommentaire(LigneAvoir la) throws DocumentException {
	
		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());		
		
		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);		
		
		cell.setPaddingLeft(5);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		cell.setPhrase(new Phrase(la.designation, fontLine));
		pptLine.addCell(cell);
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		
		if (rist) pptLine.addCell(cell);
		if (coul) pptLine.addCell(cell);		
		if (tail) pptLine.addCell(cell);
		if (unite) pptLine.addCell(cell);
		
		if (numLot) pptLine.addCell(cell);
		if (nbPieces) pptLine.addCell(cell);
		if (datePeremption) pptLine.addCell(cell);
		
		return pptLine;	
	}
		
	
	private PdfPTable makeSousTotal(double repriseSousTotal) throws DocumentException {
	
		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());		
		
		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT+PdfPCell.BOTTOM);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setPaddingBottom(6);
		cell.setPaddingTop(3);
		
		cell.setPaddingLeft(5);
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPaddingRight(3);
		cell.setPhrase(new Phrase((id.editionTTC?"Sous-total TTC":"Sous-total HT"), fontTotaux));
		pptLine.addCell(cell);
		cell.setPaddingLeft(0);	
		
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);	
		
		cell.setPhrase(new Phrase("", fontTotaux));
		pptLine.addCell(cell);
		
		if (rist) pptLine.addCell(cell);
		if (coul) pptLine.addCell(cell);
		if (tail) pptLine.addCell(cell);
		if (unite) pptLine.addCell(cell);		
		if (numLot) pptLine.addCell(cell);
		if (nbPieces) pptLine.addCell(cell);
		if (datePeremption) pptLine.addCell(cell);
		
		cell.setPhrase(new Phrase(numFormat.format(repriseSousTotal), fontTotaux));		
		pptLine.addCell(cell);
		
		return pptLine;	
	}
	
	
	private PdfPTable makeTotauxLast() throws DocumentException {
		
		PdfPTable pptTotaux = new PdfPTable(3);
		pptTotaux.setWidthPercentage(100);
		int[] widths = {50,15,35};
		pptTotaux.setWidths(widths);		
			
		PdfPCell cell = new PdfPCell(makeCadreTvaDivers());	
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.LEFT);		
		pptTotaux.addCell(cell);

		cell = new PdfPCell(pptTotaux.getDefaultCell());	
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP);		
		cell.setPhrase(new Phrase(""));
		pptTotaux.addCell(cell);

		cell = new PdfPCell(makeCadreTotaux());
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.RIGHT);		
		pptTotaux.addCell(cell);		
				
		return pptTotaux;	
	}	
		
	
	private PdfPTable makeTotauxMiddle() throws DocumentException {
		
		PdfPTable pptTotaux = new PdfPTable(3);
		pptTotaux.setWidthPercentage(100);
		int[] widths = {50,15,35};
		pptTotaux.setWidths(widths);		
			
		PdfPCell cell = new PdfPCell(pptTotaux.getDefaultCell());	
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.LEFT);
		cell.setPaddingBottom(3);	
		pptTotaux.addCell(cell);

		cell = new PdfPCell(pptTotaux.getDefaultCell());	
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP);		
		cell.setPhrase(new Phrase(""));
		pptTotaux.addCell(cell);

		cell = new PdfPCell(makeCadreSousTotal());
		cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);	
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.RIGHT);		
		pptTotaux.addCell(cell);			
				
		return pptTotaux;	
	}	
	

	private PdfPTable makeCadreTVA() {
	
		PdfPTable pptTva = new PdfPTable(3);
		pptTva.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptTva.getDefaultCell());
	
		cell.setBorder(PdfPCell.BOX);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setPhrase(new Phrase("Base HT", fontEnteteColonne));
		pptTva.addCell(cell);
		cell.setPhrase(new Phrase("Taux TVA", fontEnteteColonne));
		pptTva.addCell(cell);
		cell.setPhrase(new Phrase("Montant TVA", fontEnteteColonne));	
		pptTva.addCell(cell);
		
		Iterator<Integer> li = id.ventilTVA.keySet().iterator();
		
		while (li.hasNext()) {
			CalculVentilTVA vtva = id.ventilTVA.get(li.next());

			cell.setPaddingRight(3);
			cell.setBorder(PdfPCell.BOX-PdfPCell.TOP-PdfPCell.BOTTOM);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
			cell.setPhrase(new Phrase(numFormat.format(vtva.getMontantHT()), fontTotaux));		
			pptTva.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(vtva.getTauxTVA()) +" %"+ (vtva.getNPR()?" NPR":""), fontTotaux));
			pptTva.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(vtva.getMontantTVA()), fontTotaux));
			pptTva.addCell(cell);
		}		
		
		cell.setBorder(PdfPCell.TOP);
		cell.setColspan(3);
		cell.setPhrase(new Phrase(""));
		cell.setFixedHeight(0);
		pptTva.addCell(cell);
		
		return pptTva;
	}


	private PdfPTable makeCadreDivers() throws DocumentException {
		
		PdfPTable pptDivers = new PdfPTable(2);
		pptDivers.setWidthPercentage(100);
		int[] widths = {30,70};
		pptDivers.setWidths(widths);
		
		PdfPCell cell = new PdfPCell(pptDivers.getDefaultCell());
		
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setColspan(2);
		
		// formule pour exonration car Assujetti
	    if (id.assujetti && id.numTvaClient!="" && id.zoneUE && !id.codePaysLiv.equals("FR") && !id.regimeTVA.equals("T") && !id.typePays.equals("D")) {
	      cell.setPhrase(new Phrase("Exonration de TVA art.262 ter, I du CGI", fontModalites));// pro, num_tva, UE
	      pptDivers.addCell(cell);
	    }
	    
	    // formule pour exonration  l'International
	    if (!id.zoneUE && !id.regimeTVA.equals("T")) {
	      cell.setPhrase(new Phrase("Exonration de TVA en application de l'art.262 I du CGI", fontModalites)); //International
	      pptDivers.addCell(cell);
	    }
	    
	    // formule pour exonration DOM-TOM
	    if (id.typePays.equals("D") && !id.regimeTVA.equals("T") && (id.totalTVA==0)) {
	      cell.setPhrase(new Phrase("Exonration de TVA en application de l'art.264-2 du CGI", fontModalites));//dom
	      pptDivers.addCell(cell);
	    }
	    
		// formule pour application de TVA dans le pays de livraison
	    if (id.TVALiv && id.regimeTVA.equals("G") && (id.totalTVA!=0)) {
	      cell.setPhrase(new Phrase("Application de la TVA du pays de livraison en vertu de l'article 258 A du CGI", fontModalites));//taxe_arrivee
	      pptDivers.addCell(cell);
	    }
	    
		if (!id.numTvaClient.equals("")) {
			cell.setPhrase(new Phrase("", fontModalites));
			pptDivers.addCell(cell);
			cell.setPhrase(new Phrase("N\u00B0 TVA Client : "+ id.numTvaClient, fontModalites));
			pptDivers.addCell(cell);
		}
		
		cell.setMinimumHeight(10);
		cell.setPhrase(new Phrase("", fontModalites));
		pptDivers.addCell(cell);		
		
		return pptDivers;
	}
	
	
	private PdfPTable makeCadreTvaDivers() throws DocumentException {
		PdfPTable pptTvaDivers = new PdfPTable(1);
		pptTvaDivers.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(makeCadreTVA());	
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(3);
		cell.setPaddingTop(3);
		pptTvaDivers.addCell(cell);		
		
		cell = new PdfPCell(makeCadreDivers());
		cell.setPaddingLeft(3);
		cell.setPaddingBottom(3);	
		cell.setBorder(PdfPCell.NO_BORDER);		
		pptTvaDivers.addCell(cell);
		
		return pptTvaDivers;
	}	
	
	
	private PdfPTable makeCadreSousTotal() throws DocumentException {
	
		PdfPTable pptCadreTotaux = new PdfPTable(2);
		pptCadreTotaux.setWidthPercentage(100);
		int[] widths = {55,45};
		pptCadreTotaux.setWidths(widths);
		
		PdfPCell cell = new PdfPCell(pptCadreTotaux.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		
		cell.setBorder(PdfPCell.NO_BORDER);
		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPhrase(new Phrase((id.editionTTC?"Sous-total TTC :":"Sous-total HT :"), fontTotaux));
		pptCadreTotaux.addCell(cell);
		cell.setPhrase(new Phrase(numFormat.format(sousTotalCourant), fontTotaux));
		pptCadreTotaux.addCell(cell);
		
		cell.setColspan(2);
		cell.setPhrase(new Phrase(""));
		pptCadreTotaux.addCell(cell);
		
		return pptCadreTotaux;	
	}
	
	
	private PdfPTable makeCadreTotaux() throws DocumentException {
	
		PdfPTable pptCadreTotaux = new PdfPTable(2);
		pptCadreTotaux.setWidthPercentage(100);
		int[] widths = {55,45};
		pptCadreTotaux.setWidths(widths);
		
		PdfPCell cell = new PdfPCell(pptCadreTotaux.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);		
		cell.setBorder(PdfPCell.NO_BORDER);		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		
		if (id.tRemise>0 || (!id.sansarticles && id.fraisPort>0)) {
			cell.setPhrase(new Phrase(id.exonerationTVA?"Montant :":(id.editionTTC?"Montant TTC :":"Montant HT :"), fontTotaux));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.montantBase) +" \u20AC", fontTotaux));
			pptCadreTotaux.addCell(cell);

			if (id.tRemise>0) {		
				cell.setPhrase(new Phrase("Remise ("+ numFormatCap.format(id.tRemise) +" %) :", fontTotaux));
				pptCadreTotaux.addCell(cell);
				cell.setPhrase(new Phrase(""+ numFormat.format(id.mRemise) +" \u20AC", fontTotaux));
				pptCadreTotaux.addCell(cell);
			}

			if (!id.sansarticles && id.fraisPort>0) {
				cell.setPhrase(new Phrase("Frais de livraison :", fontTotaux));
				pptCadreTotaux.addCell(cell);
				cell.setPhrase(new Phrase(numFormat.format(id.fraisPort) +" \u20AC", fontTotaux));
				pptCadreTotaux.addCell(cell);
				
				if (id.tRemiseFP>0) {		
					cell.setPhrase(new Phrase("Remise sur frais de port ("+ numFormatCap.format(id.tRemiseFP) +" %) :", fontTotaux));
					pptCadreTotaux.addCell(cell);
					cell.setPhrase(new Phrase(""+ numFormat.format(id.mRemiseFP) +" \u20AC", fontTotaux));
					pptCadreTotaux.addCell(cell);
				}
			}
		}

		cell.setPhrase(new Phrase(id.exonerationTVA?"Total :":(id.editionTTC?"Total TTC :":"Total HT :"),id.editionTTC || id.exonerationTVA?fontTTC:fontTotaux));
		pptCadreTotaux.addCell(cell);
		cell.setPhrase(new Phrase(numFormat.format(id.totalBase) +" \u20AC",id.editionTTC || id.exonerationTVA?fontTTC:fontTotaux));
		pptCadreTotaux.addCell(cell);

		if (!id.exonerationTVA) {
			cell.setPhrase(new Phrase((id.editionTTC?"Dont TVA :":"Total TVA :"), fontTotaux));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.totalTVA) +" \u20AC", fontTotaux));
			pptCadreTotaux.addCell(cell);
			
			if (!id.editionTTC) {
				cell.setPhrase(new Phrase("Total TTC :", fontTTC));
				pptCadreTotaux.addCell(cell);
				cell.setPhrase(new Phrase(numFormat.format(id.montantTTC) +" \u20AC", fontTTC));
				pptCadreTotaux.addCell(cell);		
			}
		}

		if (id.tEscompte>0) {
			cell.setPhrase(new Phrase("", fontTotaux));
			cell.setColspan(2);
			cell.setFixedHeight(10);
			pptCadreTotaux.addCell(cell);
			cell.setColspan(1);
			cell.setMinimumHeight(0);
			
			cell.setPhrase(new Phrase("Escompte ("+ numFormatCap.format(id.tEscompte) +" %) :", fontTotaux));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.mEscompte) +" \u20AC", fontTotaux));
			pptCadreTotaux.addCell(cell);

			cell.setPhrase(new Phrase("Net \u00E0 payer :", fontTTC));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.netAPayer) +" \u20AC", fontTTC));
			pptCadreTotaux.addCell(cell);	
		}

		cell.setColspan(2);
		cell.setPhrase(new Phrase(""));
		pptCadreTotaux.addCell(cell);
		
		return pptCadreTotaux;
	}


	private PdfPTable makeZoneCommentaire() throws DocumentException {		
		
		PdfPTable pptCom = new PdfPTable(nbCols);
		pptCom.setWidthPercentage(100);
		pptCom.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptCom.getDefaultCell());
		cell.setBorder(PdfPCell.LEFT + PdfPCell.RIGHT);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);		
		cell.setPaddingLeft(5);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase("", fontLine));
		pptCom.addCell(cell);
		cell.setPhrase(new Phrase(id.comFin + id.mentions, fontLine));
		pptCom.addCell(cell);
		cell.setPhrase(new Phrase("", fontLine));
		pptCom.addCell(cell);
		pptCom.addCell(cell);
		pptCom.addCell(cell);
		
		if (rist) pptCom.addCell(cell);
		if (coul) pptCom.addCell(cell);		
		if (tail) pptCom.addCell(cell);	
		if (unite) pptCom.addCell(cell);	
		
		if (numLot) pptCom.addCell(cell);
		if (nbPieces) pptCom.addCell(cell);
		if (datePeremption) pptCom.addCell(cell);
	
		return pptCom;		
	}
	
	
	private PdfPTable makeInfosLegales() {
	
		PdfPTable pptLegales = new PdfPTable(1);
		pptLegales.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptLegales.getDefaultCell());		
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPhrase(new Phrase(id.raisonSociale +" - "+ id.typeSociete + (id.capital>0?" au capital de "+ numFormatCap.format(id.capital) +" Euros":"") +" - immatricul\u00E9e au RCS "+ id.rcs +" "+ id.siren +" - N\u00B0 TVA : "+ id.tvaIntra, fontLegale));
		
		pptLegales.addCell(cell);
		
		return pptLegales;	
	}
	

} // fin AvoirStd
